import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function seedDefaultCatalog() {
    try {
        console.log('🌱 Seeding default catalog...');

        // Check if default catalog already exists
        const existingCatalog = await prisma.catalog.findFirst({
            where: { isDefault: true },
        });

        if (existingCatalog) {
            console.log('✅ Default catalog already exists:', existingCatalog.name);
            return existingCatalog;
        }

        // Create default catalog
        const defaultCatalog = await prisma.catalog.create({
            data: {
                name: 'Main Catalog',
                slug: 'main',
                description: 'Default product catalog for all existing products',
                isActive: true,
                isDefault: true,
            },
        });

        console.log('✅ Created default catalog:', defaultCatalog.name);

        // Update all existing categories to belong to default catalog
        const categoriesUpdated = await prisma.category.updateMany({
            where: {
                catalogId: null,
            },
            data: {
                catalogId: defaultCatalog.id,
            },
        });

        console.log(`✅ Updated ${categoriesUpdated.count} categories to default catalog`);

        // Update all existing products to belong to default catalog
        const productsUpdated = await prisma.product.updateMany({
            where: {
                catalogId: null,
            },
            data: {
                catalogId: defaultCatalog.id,
            },
        });

        console.log(`✅ Updated ${productsUpdated.count} products to default catalog`);

        // Update all existing banner slides to belong to default catalog
        const bannersUpdated = await prisma.bannerSlide.updateMany({
            where: {
                catalogId: null,
            },
            data: {
                catalogId: defaultCatalog.id,
            },
        });

        console.log(`✅ Updated ${bannersUpdated.count} banner slides to default catalog`);

        // Assign all existing customers to default catalog
        const customers = await prisma.user.findMany({
            where: { role: 'CUSTOMER' },
        });

        for (const customer of customers) {
            // Check if assignment already exists
            const existingAssignment = await prisma.catalogCustomer.findUnique({
                where: {
                    catalogId_customerId: {
                        catalogId: defaultCatalog.id,
                        customerId: customer.id,
                    },
                },
            });

            if (!existingAssignment) {
                await prisma.catalogCustomer.create({
                    data: {
                        catalogId: defaultCatalog.id,
                        customerId: customer.id,
                        canOrder: true,
                        canQuote: true,
                    },
                });
            }
        }

        console.log(`✅ Assigned ${customers.length} customers to default catalog`);

        console.log('\n🎉 Default catalog seeding complete!');
        console.log('📊 Summary:');
        console.log(`   - Catalog: ${defaultCatalog.name} (${defaultCatalog.slug})`);
        console.log(`   - Categories: ${categoriesUpdated.count}`);
        console.log(`   - Products: ${productsUpdated.count}`);
        console.log(`   - Banner Slides: ${bannersUpdated.count}`);
        console.log(`   - Customers: ${customers.length}`);

        return defaultCatalog;
    } catch (error) {
        console.error('❌ Error seeding default catalog:', error);
        throw error;
    } finally {
        await prisma.$disconnect();
    }
}

// Run if called directly
if (require.main === module) {
    seedDefaultCatalog()
        .then(() => {
            console.log('\n✅ Seed script completed successfully');
            process.exit(0);
        })
        .catch((error) => {
            console.error('\n❌ Seed script failed:', error);
            process.exit(1);
        });
}

export default seedDefaultCatalog;
