import { buildAuthHeaders } from '../utils/apiClient';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';


export interface BannerSlide {
  id: string;
  title?: string;
  subtitle?: string;
  buttonText?: string;
  slideType: 'IMAGE' | 'TEXT';
  imageUrl?: string;
  backgroundColor?: string;
  textColor?: string;
  linkType?: string;
  linkedProductId?: string;
  linkedCategoryId?: string;
  linkedProduct?: {
    id: string;
    name: string;
    slug: string;
  };
  linkedCategory?: {
    id: string;
    name: string;
    slug: string;
  };
  displayOrder: number;
  isActive: boolean;
  duration?: number;
  createdAt: string;
  updatedAt: string;
}

export interface CreateBannerSlideData {
  title?: string;
  subtitle?: string;
  buttonText?: string;
  slideType: 'IMAGE' | 'TEXT';
  imageUrl?: string;
  backgroundColor?: string;
  textColor?: string;
  linkType?: string;
  linkedProductId?: string;
  linkedCategoryId?: string;
  displayOrder?: number;
  isActive?: boolean;
  duration?: number;
}

class BannerService {


  async getAllSlides(catalogId?: string | null): Promise<BannerSlide[]> {
    const response = await fetch(`${API_URL}/banners/admin/all`, {
      headers: buildAuthHeaders({ catalogId }),
    });


    if (!response.ok) {
      throw new Error('Failed to fetch banner slides');
    }

    const data = await response.json();
    return data.data;
  }

  async getSlide(id: string, catalogId?: string | null): Promise<BannerSlide> {
    const response = await fetch(`${API_URL}/banners/${id}`, {
      headers: buildAuthHeaders({ catalogId }),
    });


    if (!response.ok) {
      throw new Error('Failed to fetch banner slide');
    }

    const data = await response.json();
    return data.data;
  }

  async createSlide(slideData: CreateBannerSlideData, catalogId?: string | null): Promise<BannerSlide> {
    const response = await fetch(`${API_URL}/banners`, {
      method: 'POST',
      headers: buildAuthHeaders({ catalogId }),
      body: JSON.stringify(slideData),

    });

    if (!response.ok) {
      throw new Error('Failed to create banner slide');
    }

    const data = await response.json();
    return data.data;
  }

  async updateSlide(id: string, slideData: Partial<CreateBannerSlideData>, catalogId?: string | null): Promise<BannerSlide> {
    const response = await fetch(`${API_URL}/banners/${id}`, {
      method: 'PUT',
      headers: buildAuthHeaders({ catalogId }),
      body: JSON.stringify(slideData),

    });

    if (!response.ok) {
      throw new Error('Failed to update banner slide');
    }

    const data = await response.json();
    return data.data;
  }

  async deleteSlide(id: string, catalogId?: string | null): Promise<void> {
    const response = await fetch(`${API_URL}/banners/${id}`, {
      method: 'DELETE',
      headers: buildAuthHeaders({ catalogId }),
    });


    if (!response.ok) {
      throw new Error('Failed to delete banner slide');
    }
  }

  async reorderSlides(slides: { id: string; displayOrder: number }[], catalogId?: string | null): Promise<void> {
    const response = await fetch(`${API_URL}/banners/reorder`, {
      method: 'POST',
      headers: buildAuthHeaders({ catalogId }),
      body: JSON.stringify({ slides }),

    });

    if (!response.ok) {
      throw new Error('Failed to reorder banner slides');
    }
  }
}

export const bannerService = new BannerService();
