import { Plus } from 'lucide-react';
import { Product } from '../types/api';
import { useCart } from '../context/CartContext';

interface ProductCardProps {
  product: Product;
  onAddToCart: (product: Product) => void;
  onClick: () => void;
}

export default function ProductCard({ product, onAddToCart, onClick }: ProductCardProps) {
  const { getCartQuantity } = useCart();
  const cartQuantity = getCartQuantity(product.id);
  const availableStock = product.stock - cartQuantity;

  const handleAddClick = (e: React.MouseEvent) => {
    e.stopPropagation();
    if (availableStock > 0) {
      onAddToCart(product);
    }
  };

  return (
    <div
      onClick={onClick}
      className="bg-white rounded-lg shadow-sm overflow-hidden hover:shadow-md transition-shadow cursor-pointer"
    >
      <div className="aspect-square bg-gray-100 overflow-hidden">
        <img
          src={product.image}
          alt={product.name}
          className="w-full h-full object-cover"
        />
      </div>

      <div className="p-3">
        <p className="text-xs text-gray-500 mb-1">{product.brand}</p>
        <h3 className="font-medium text-gray-800 text-sm mb-2 line-clamp-2">{product.name}</h3>

        <div className="flex items-end justify-between gap-3">
          <div className="flex-1 min-w-0">
            {/* Calculate discount if active */}
            {(() => {
              const now = new Date();
              const isDiscountActive = product.hasDiscount &&
                product.discountValue &&
                product.discountValue > 0 &&
                (!product.discountStartDate || new Date(product.discountStartDate) <= now) &&
                (!product.discountEndDate || new Date(product.discountEndDate) >= now);

              const basePrice = product.basePrice || product.price || 0;
              let discountAmount = 0;
              let finalPrice = basePrice;

              if (isDiscountActive) {
                if (product.discountType === 'percentage') {
                  discountAmount = (basePrice * product.discountValue!) / 100;
                } else if (product.discountType === 'fixed') {
                  discountAmount = Math.min(product.discountValue!, basePrice);
                }
                finalPrice = Math.max(0, basePrice - discountAmount);
              }

              return (
                <div className="flex flex-col">
                  {isDiscountActive ? (
                    <>
                      {/* Discount Badge & Old Price */}
                      <div className="flex items-center gap-2 mb-1 flex-wrap">
                        <span className="px-1.5 py-0.5 bg-red-500 text-white text-[10px] rounded font-bold whitespace-nowrap">
                          {product.discountType === 'percentage'
                            ? `-${product.discountValue}%`
                            : `-${product.discountValue?.toFixed(0)} DT`}
                        </span>
                        <span className="line-through text-gray-400 text-xs">
                          {basePrice.toFixed(2)}
                        </span>
                      </div>

                      {/* Final Price */}
                      <div className="flex items-baseline gap-1 flex-wrap">
                        <span className="text-lg font-bold text-green-600 leading-none">
                          {finalPrice.toFixed(2)} <span className="text-xs font-normal text-gray-500">HT</span>
                        </span>
                      </div>

                      {/* TTC Price */}
                      {!product.tvaExempt && (
                        <p className="text-[10px] text-gray-500 mt-0.5">
                          {(finalPrice * (1 + (product.tvaRate || 19) / 100)).toFixed(2)} TTC
                        </p>
                      )}
                    </>
                  ) : (
                    <>
                      {/* No Discount - Normal Display */}
                      <div className="flex items-baseline gap-1 flex-wrap">
                        <span className="text-lg font-bold text-gray-900 leading-none">
                          {basePrice.toFixed(2)} <span className="text-xs font-normal text-gray-500">HT</span>
                        </span>
                      </div>
                      {!product.tvaExempt && (
                        <p className="text-[10px] text-gray-500 mt-0.5">
                          {(basePrice * (1 + (product.tvaRate || 19) / 100)).toFixed(2)} TTC
                        </p>
                      )}
                    </>
                  )}
                </div>
              );
            })()}

            <p className="text-[10px] text-gray-400 mt-1 truncate">
              {availableStock > 0 ? (
                <span className="text-green-600">{availableStock} in stock</span>
              ) : (
                <span className="text-red-500">Out of stock</span>
              )}
            </p>
          </div>

          <button
            onClick={handleAddClick}
            disabled={availableStock <= 0}
            className={`shrink-0 w-10 h-10 flex items-center justify-center rounded-lg transition-colors ${availableStock > 0
              ? 'bg-blue-600 hover:bg-blue-700 text-white shadow-sm'
              : 'bg-gray-100 text-gray-400 cursor-not-allowed'
              }`}
            title="Add to Cart"
          >
            <Plus className="w-5 h-5" />
          </button>
        </div>
      </div>
    </div>
  );
}
