import { useState, useEffect, useRef } from 'react';
import { Search, X, ChevronRight, Package, Folder } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import { API } from '../services/api';
import { Product, Category } from '../types/api';
import { useLanguage } from '../contexts/LanguageContext';

interface GlobalSearchProps {
    categories: Category[];
    onClose?: () => void;
}

export default function GlobalSearch({ categories, onClose }: GlobalSearchProps) {
    const { t } = useLanguage();
    const navigate = useNavigate();
    const [query, setQuery] = useState('');
    const [results, setResults] = useState<{
        products: Product[];
        categories: Category[];
    }>({ products: [], categories: [] });
    const [loading, setLoading] = useState(false);
    const [showResults, setShowResults] = useState(false);
    const searchRef = useRef<HTMLDivElement>(null);

    // Debounce search
    useEffect(() => {
        const timer = setTimeout(() => {
            if (query.trim().length >= 2) {
                performSearch(query);
            } else {
                setResults({ products: [], categories: [] });
                setShowResults(false);
            }
        }, 300);

        return () => clearTimeout(timer);
    }, [query]);

    // Close on click outside
    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (searchRef.current && !searchRef.current.contains(event.target as Node)) {
                setShowResults(false);
            }
        };

        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
    }, []);

    const performSearch = async (searchQuery: string) => {
        setLoading(true);
        try {
            // Search products via API
            const productResponse = await API.searchProducts(searchQuery, 1, 5);

            // Search categories client-side (flattened list would be better, but we'll search top-level and children)
            const searchCategories = (cats: Category[], term: string): Category[] => {
                let matches: Category[] = [];
                for (const cat of cats) {
                    if (cat.name.toLowerCase().includes(term.toLowerCase())) {
                        matches.push(cat);
                    }
                    if (cat.children) {
                        matches = [...matches, ...searchCategories(cat.children, term)];
                    }
                }
                return matches;
            };

            const matchedCategories = searchCategories(categories, searchQuery).slice(0, 3);

            setResults({
                products: productResponse.products || [],
                categories: matchedCategories
            });
            setShowResults(true);
        } catch (error) {
            console.error('Search error:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleProductClick = (productId: string) => {
        navigate(`/products/${productId}`);
        setShowResults(false);
        setQuery('');
        if (onClose) onClose();
    };

    const handleCategoryClick = (categoryId: string) => {
        navigate(`/categories/${categoryId}`);
        setShowResults(false);
        setQuery('');
        if (onClose) onClose();
    };

    return (
        <div ref={searchRef} className="relative w-full max-w-2xl mx-auto">
            <div className="relative">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
                <input
                    type="text"
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    placeholder={t.home?.searchPlaceholder || "Search products, categories..."}
                    className="w-full pl-10 pr-10 py-2.5 bg-gray-50 border border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                />
                {query && (
                    <button
                        onClick={() => {
                            setQuery('');
                            setShowResults(false);
                        }}
                        className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 hover:text-gray-600"
                    >
                        <X className="w-4 h-4" />
                    </button>
                )}
            </div>

            {/* Results Dropdown */}
            {showResults && (
                <div className="absolute top-full left-0 right-0 mt-2 bg-white rounded-xl shadow-xl border border-gray-100 max-h-[80vh] overflow-y-auto z-50">
                    {loading ? (
                        <div className="p-4 text-center text-gray-500">Searching...</div>
                    ) : (
                        <>
                            {results.categories.length > 0 && (
                                <div className="p-2">
                                    <h3 className="text-xs font-semibold text-gray-500 uppercase tracking-wider px-3 py-2">
                                        Categories
                                    </h3>
                                    {results.categories.map((cat) => (
                                        <button
                                            key={cat.id}
                                            onClick={() => handleCategoryClick(cat.id)}
                                            className="w-full flex items-center gap-3 px-3 py-2 hover:bg-gray-50 rounded-lg transition-colors text-left"
                                        >
                                            <div className="w-8 h-8 bg-blue-50 rounded-lg flex items-center justify-center flex-shrink-0">
                                                <Folder className="w-4 h-4 text-blue-600" />
                                            </div>
                                            <span className="font-medium text-gray-900">{cat.name}</span>
                                            <ChevronRight className="w-4 h-4 text-gray-400 ml-auto" />
                                        </button>
                                    ))}
                                </div>
                            )}

                            {results.products.length > 0 && (
                                <div className="p-2 border-t border-gray-100">
                                    <h3 className="text-xs font-semibold text-gray-500 uppercase tracking-wider px-3 py-2">
                                        Products
                                    </h3>
                                    {results.products.map((product) => (
                                        <button
                                            key={product.id}
                                            onClick={() => handleProductClick(product.id)}
                                            className="w-full flex items-center gap-3 px-3 py-2 hover:bg-gray-50 rounded-lg transition-colors text-left"
                                        >
                                            <div className="w-10 h-10 bg-gray-100 rounded-lg overflow-hidden flex-shrink-0">
                                                {product.image ? (
                                                    <img src={product.image} alt={product.name} className="w-full h-full object-cover" />
                                                ) : (
                                                    <div className="w-full h-full flex items-center justify-center">
                                                        <Package className="w-5 h-5 text-gray-400" />
                                                    </div>
                                                )}
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <p className="font-medium text-gray-900 truncate">{product.name}</p>
                                                <p className="text-sm text-gray-500 truncate">{product.brand}</p>
                                            </div>
                                            {product.stock > 0 ? (
                                                <span className="text-xs font-medium text-green-600 bg-green-50 px-2 py-1 rounded-full whitespace-nowrap">
                                                    In Stock
                                                </span>
                                            ) : (
                                                <span className="text-xs font-medium text-red-600 bg-red-50 px-2 py-1 rounded-full whitespace-nowrap">
                                                    Out of Stock
                                                </span>
                                            )}
                                        </button>
                                    ))}
                                </div>
                            )}

                            {results.categories.length === 0 && results.products.length === 0 && (
                                <div className="p-8 text-center text-gray-500">
                                    No results found for "{query}"
                                </div>
                            )}
                        </>
                    )}
                </div>
            )}
        </div>
    );
}
