import { Response } from 'express';
import { PrismaClient, QuotationStatus, PaymentTerm } from '@prisma/client';
import { resolveCatalogContext } from '../utils/catalogContext.js';
import { AuthRequest } from '../middleware/auth.js';
import {
  calculateLineItemTva,
  calculateDocumentTva,
  getEffectiveTvaRate,
  isCustomerTvaExempt
} from '../utils/tvaCalculations.js';


const prisma = new PrismaClient();

function generateQuotationNumber(): string {
  const timestamp = Date.now().toString(36).toUpperCase();
  const random = Math.random().toString(36).substring(2, 6).toUpperCase();
  return `QUO-${timestamp}-${random}`;
}

function generateOrderNumber(): string {
  const timestamp = Date.now().toString(36).toUpperCase();
  const random = Math.random().toString(36).substring(2, 6).toUpperCase();
  return `ORD-${timestamp}-${random}`;
}

function calculateDueDate(paymentTerm?: PaymentTerm | null): Date | null {
  if (!paymentTerm || paymentTerm === 'IMMEDIATE') return null;
  const daysMap = { NET_30: 30, NET_60: 60, NET_90: 90, NET_120: 120 };
  const days = daysMap[paymentTerm] || 0;
  const dueDate = new Date();
  dueDate.setDate(dueDate.getDate() + days);
  return dueDate;
}

export const createQuotation = async (req: AuthRequest, res: Response) => {
  try {
    const { items, shippingAddressId, shippingAddress, notes } = req.body;
    const userId = req.user?.id;
    const catalog = await resolveCatalogContext(req, prisma);

    if (!userId) {
      return res.status(401).json({ success: false, error: { message: 'Unauthorized' } });
    }

    const user = await prisma.user.findUnique({ where: { id: userId } });
    if (!user) {
      return res.status(404).json({ success: false, error: { message: 'User not found' } });
    }

    if (!items?.length) {
      return res.status(400).json({ success: false, error: { message: 'Items required' } });
    }

    // Check if customer is TVA exempt
    const customerTvaExempt = await isCustomerTvaExempt(userId);

    const quotationItems = [];
    const lineItemResults = [];

    for (const item of items) {
      const product = await prisma.product.findUnique({ where: { id: item.productId } });
      if (!product) {
        return res.status(404).json({ success: false, error: { message: `Product ${item.productId} not found` } });
      }

      if (product.catalogId !== catalog.id) {
        return res.status(400).json({ success: false, error: { message: `Product ${product.name} does not belong to the current catalog` } });
      }

      // Get effective TVA rate for this product in this catalog
      const tvaInfo = await getEffectiveTvaRate(product.id, catalog.id);
      const effectiveTvaExempt = customerTvaExempt || tvaInfo.exempt;

      const unitPrice = item.unitPrice || product.basePrice;

      // Calculate product discount if active
      let discountAmount = 0;
      const now = new Date();
      const isDiscountActive = product.hasDiscount &&
        product.discountValue &&
        Number(product.discountValue) > 0 &&
        (!product.discountStartDate || new Date(product.discountStartDate) <= now) &&
        (!product.discountEndDate || new Date(product.discountEndDate) >= now);

      if (isDiscountActive) {
        const baseAmount = Number(unitPrice) * item.quantity;
        if (product.discountType === 'percentage') {
          discountAmount = (baseAmount * Number(product.discountValue)) / 100;
        } else if (product.discountType === 'fixed') {
          // Fixed discount per unit, multiplied by quantity
          discountAmount = Math.min(Number(product.discountValue) * item.quantity, baseAmount);
        }
      }

      // Calculate line item TVA (discount is applied before TVA)
      const lineCalc = calculateLineItemTva({
        quantity: item.quantity,
        unitPrice: Number(unitPrice),
        discount: discountAmount, // Use calculated product discount
        tvaRate: tvaInfo.rate,
        tvaExempt: effectiveTvaExempt,
      });

      lineItemResults.push(lineCalc);

      quotationItems.push({
        productId: product.id,
        productName: product.name,
        productSku: product.sku,
        selectedSize: item.selectedSize,
        selectedUnitType: item.selectedUnitType,
        quantity: item.quantity,
        unitPrice,
        discount: lineCalc.discount,
        totalHT: lineCalc.totalNetHT,
        tvaRate: lineCalc.tvaRate,
        tvaAmount: lineCalc.tvaAmount,
        totalTTC: lineCalc.totalTTC,
        tvaExempt: lineCalc.tvaExempt,
        totalPrice: lineCalc.totalTTC, // Backward compatibility
      });
    }

    // Calculate document-level TVA summary
    const tvaSummary = calculateDocumentTva(lineItemResults);
    const anticipatedOutstanding = Number(user.currentOutstanding || 0) + tvaSummary.netAPayer;

    const quotation = await prisma.quotation.create({
      data: {
        quotationNumber: generateQuotationNumber(),
        userId,
        status: 'DRAFT',

        // New TVA fields
        totalBrutHT: tvaSummary.totalBrutHT,
        totalDiscount: tvaSummary.totalDiscount,
        totalNetHT: tvaSummary.totalNetHT,
        tva19Amount: tvaSummary.tva19Amount,
        tva7Amount: tvaSummary.tva7Amount,
        tva0Amount: tvaSummary.tva0Amount,
        totalTVA: tvaSummary.totalTVA,
        netAPayer: tvaSummary.netAPayer,
        netAPayerWords: tvaSummary.netAPayerWords,
        customerTvaExempt,

        // Deprecated (backward compatibility)
        subtotal: tvaSummary.totalNetHT,
        taxAmount: tvaSummary.totalTVA,
        totalAmount: tvaSummary.netAPayer,
        anticipatedOutstanding,

        shippingAddressId,
        shippingAddress,
        notes,
        items: { create: quotationItems },
      },
      include: {
        items: { include: { product: true } },
        user: true,
      },
    });

    res.status(201).json({ success: true, data: quotation });
  } catch (error) {
    console.error('Error creating quotation:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to create quotation' } });
  }
};

export const getAllQuotations = async (req: AuthRequest, res: Response) => {
  try {
    const userId = req.user?.id;
    const userRole = req.user?.role;
    const { status, customerId, page = '1', limit = '20' } = req.query;

    const pageNum = parseInt(page as string);
    const limitNum = parseInt(limit as string);
    const skip = (pageNum - 1) * limitNum;

    const where: any = {};

    // For customers, filter by catalog
    if (userRole === 'CUSTOMER') {
      const catalog = await resolveCatalogContext(req, prisma);
      where.items = {
        some: {
          product: {
            catalogId: catalog.id
          }
        }
      };
      where.userId = userId;
    }

    // For admins/managers, optionally filter by catalog if provided
    if (userRole === 'ADMIN' || userRole === 'MANAGER') {
      // Try to get catalog context, but don't require it
      try {
        const catalog = await resolveCatalogContext(req, prisma, { required: false });
        if (catalog) {
          where.items = {
            some: {
              product: {
                catalogId: catalog.id
              }
            }
          };
        }
      } catch (error) {
        // Ignore catalog resolution errors for admins - they can see all catalogs
      }

      if (customerId) {
        where.userId = customerId;
      }
    }

    if (status) {
      where.status = status;
    }

    const [quotations, total] = await Promise.all([
      prisma.quotation.findMany({
        where,
        include: {
          items: { include: { product: true } },
          user: true,
          reviewer: true,
        },
        orderBy: { createdAt: 'desc' },
        skip,
        take: limitNum,
      }),
      prisma.quotation.count({ where }),
    ]);

    res.json({
      success: true,
      data: quotations,
      pagination: {
        page: pageNum,
        limit: limitNum,
        total,
        totalPages: Math.ceil(total / limitNum),
      },
    });
  } catch (error) {
    console.error('Error fetching quotations:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to fetch quotations' } });
  }
};

export const getQuotationById = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id;
    const userRole = req.user?.role;

    const quotation = await prisma.quotation.findUnique({
      where: { id },
      include: {
        items: { include: { product: { include: { images: { take: 1 } } } } },
        user: true,
        shippingAddr: true,
        reviewer: true,
        convertedOrder: true,
      },
    });

    if (!quotation) {
      return res.status(404).json({ success: false, error: { message: 'Quotation not found' } });
    }

    if (userRole === 'CUSTOMER' && quotation.userId !== userId) {
      return res.status(403).json({ success: false, error: { message: 'Forbidden' } });
    }

    res.json({ success: true, data: quotation });
  } catch (error) {
    console.error('Error fetching quotation:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to fetch quotation' } });
  }
};

export const submitQuotation = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id;

    const quotation = await prisma.quotation.findUnique({ where: { id } });
    if (!quotation) {
      return res.status(404).json({ success: false, error: { message: 'Quotation not found' } });
    }

    if (quotation.userId !== userId) {
      return res.status(403).json({ success: false, error: { message: 'Forbidden' } });
    }

    if (quotation.status !== 'DRAFT') {
      return res.status(400).json({ success: false, error: { message: 'Only drafts can be submitted' } });
    }

    const updated = await prisma.quotation.update({
      where: { id },
      data: { status: 'PENDING_APPROVAL' },
      include: { items: true, user: true },
    });

    res.json({ success: true, data: updated, message: 'Quotation submitted for approval' });
  } catch (error) {
    console.error('Error submitting quotation:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to submit quotation' } });
  }
};

/**
 * Update quotation (Admin only)
 * Allows admin to modify quantities or remove items before approving
 */
export const updateQuotation = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const { items, notes } = req.body; // Changed from adminNotes to notes
    const userRole = req.user?.role;
    const adminUserId = req.user?.id;

    // Only admin/manager can update quotations
    if (userRole !== 'ADMIN' && userRole !== 'MANAGER') {
      return res.status(403).json({ success: false, error: { message: 'Forbidden' } });
    }

    // Fetch existing quotation
    const quotation = await prisma.quotation.findUnique({
      where: { id },
      include: { items: true, user: true },
    });

    if (!quotation) {
      return res.status(404).json({ success: false, error: { message: 'Quotation not found' } });
    }

    // Only allow editing DRAFT or PENDING_APPROVAL quotations
    if (quotation.status !== 'DRAFT' && quotation.status !== 'PENDING_APPROVAL') {
      return res.status(400).json({
        success: false,
        error: { message: 'Only draft or pending quotations can be edited' }
      });
    }

    const catalog = await resolveCatalogContext(req, prisma);
    const customer = quotation.user;
    const customerTvaExempt = await isCustomerTvaExempt(customer.id);

    // Process item updates
    const updatedItemsData: Array<{
      id: string;
      quantity: number;
      totalHT: number;
      tvaAmount: number;
      totalTTC: number;
    }> = [];
    const itemsToDelete: string[] = [];

    for (const itemUpdate of items) {
      if (itemUpdate.delete) {
        itemsToDelete.push(itemUpdate.id);
      } else {
        const existingItem = quotation.items.find(i => i.id === itemUpdate.id);
        if (existingItem) {
          // Fetch product for TVA rate
          const product = await prisma.product.findUnique({
            where: { id: existingItem.productId },
          });

          if (!product) continue;

          // Get effective TVA rate
          const tvaInfo = await getEffectiveTvaRate(product.id, catalog.id);
          const effectiveTvaExempt = customerTvaExempt || tvaInfo.exempt;
          const effectiveTvaRate = effectiveTvaExempt ? 0 : tvaInfo.rate;

          // Calculate line item TVA with new quantity
          const lineCalc = calculateLineItemTva({
            quantity: itemUpdate.quantity || existingItem.quantity,
            unitPrice: Number(existingItem.unitPrice),
            discount: Number(existingItem.discount || 0),
            tvaRate: effectiveTvaRate,
            tvaExempt: effectiveTvaExempt,
          });

          updatedItemsData.push({
            id: existingItem.id,
            quantity: itemUpdate.quantity || existingItem.quantity,
            totalHT: lineCalc.totalNetHT,
            tvaAmount: lineCalc.tvaAmount,
            totalTTC: lineCalc.totalTTC,
          });
        }
      }
    }

    // Calculate document totals from remaining items
    const remainingItems = quotation.items.filter(item => !itemsToDelete.includes(item.id));
    const lineItemResults = remainingItems.map(item => {
      const updated = updatedItemsData.find(u => u.id === item.id);
      if (updated) {
        return {
          totalBrutHT: updated.quantity * Number(item.unitPrice),
          discount: Number(item.discount || 0),
          totalNetHT: updated.totalHT,
          tvaRate: Number(item.tvaRate),
          tvaAmount: updated.tvaAmount,
          totalTTC: updated.totalTTC,
          tvaExempt: item.tvaExempt,
        };
      } else {
        return {
          totalBrutHT: Number(item.totalHT) + Number(item.discount || 0),
          discount: Number(item.discount || 0),
          totalNetHT: Number(item.totalHT),
          tvaRate: Number(item.tvaRate),
          tvaAmount: Number(item.tvaAmount),
          totalTTC: Number(item.totalTTC),
          tvaExempt: item.tvaExempt,
        };
      }
    });

    const documentTotals = calculateDocumentTva(lineItemResults);

    // Update quotation in transaction
    const updatedQuotation = await prisma.$transaction(async (tx) => {
      // Delete removed items
      if (itemsToDelete.length > 0) {
        await tx.quotationItem.deleteMany({
          where: { id: { in: itemsToDelete } },
        });
      }

      // Update modified items
      for (const itemUpdate of updatedItemsData) {
        await tx.quotationItem.update({
          where: { id: itemUpdate.id },
          data: {
            quantity: itemUpdate.quantity,
            totalHT: itemUpdate.totalHT,
            tvaAmount: itemUpdate.tvaAmount,
            totalTTC: itemUpdate.totalTTC,
            totalPrice: itemUpdate.totalTTC, // Deprecated field
          },
        });
      }

      // Update quotation totals
      const updated = await tx.quotation.update({
        where: { id },
        data: {
          // TVA fields
          totalBrutHT: documentTotals.totalBrutHT,
          totalDiscount: documentTotals.totalDiscount,
          totalNetHT: documentTotals.totalNetHT,
          tva19Amount: documentTotals.tva19Amount,
          tva7Amount: documentTotals.tva7Amount,
          tva0Amount: documentTotals.tva0Amount,
          totalTVA: documentTotals.totalTVA,
          netAPayer: documentTotals.netAPayer,
          netAPayerWords: documentTotals.netAPayerWords,

          // Deprecated fields (for backward compatibility)
          subtotal: documentTotals.totalNetHT,
          taxAmount: documentTotals.totalTVA,
          totalAmount: documentTotals.netAPayer,

          // Update anticipated outstanding for credit check
          anticipatedOutstanding: Number(customer.currentOutstanding || 0) + documentTotals.netAPayer,

          // Update notes if provided
          notes: notes || quotation.notes,
          reviewedBy: adminUserId,
          reviewedAt: new Date(),
        },
        include: {
          items: {
            include: { product: true },
          },
          user: true,
          shippingAddr: true,
        },
      });

      return updated;
    });

    res.json({
      success: true,
      message: 'Quotation updated successfully',
      data: updatedQuotation,
    });
  } catch (error) {
    console.error('Error updating quotation:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to update quotation' } });
  }
};

export const approveQuotation = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const adminUserId = req.user?.id;
    const userRole = req.user?.role;

    if (userRole !== 'ADMIN' && userRole !== 'MANAGER') {
      return res.status(403).json({ success: false, error: { message: 'Forbidden' } });
    }

    const quotation = await prisma.quotation.findUnique({
      where: { id },
      include: { user: true, items: { include: { product: true } }, shippingAddr: true },
    });

    if (!quotation) {
      return res.status(404).json({ success: false, error: { message: 'Quotation not found' } });
    }

    // Allow approval of DRAFT or PENDING_APPROVAL quotations
    if (quotation.status !== 'PENDING_APPROVAL' && quotation.status !== 'DRAFT') {
      return res.status(400).json({
        success: false,
        error: { message: 'Only draft or pending quotations can be approved' }
      });
    }

    const customer = quotation.user;
    const creditWarning = Number(quotation.anticipatedOutstanding) > Number(customer.financialLimit || 0);

    const result = await prisma.$transaction(async (tx) => {
      const order = await tx.order.create({
        data: {
          orderNumber: generateOrderNumber(),
          userId: customer.id,
          customerName: `${customer.firstName || ''} ${customer.lastName || ''}`.trim() || customer.email,
          customerEmail: customer.email,
          customerPhone: customer.phone || '',
          shippingAddressId: quotation.shippingAddressId,
          shippingAddress: quotation.shippingAddress,
          status: 'PENDING',
          paymentMethod: 'NET_TERMS',
          paymentStatus: 'PENDING',
          paymentTerm: customer.paymentTerm || 'NET_30',
          dueDate: calculateDueDate(customer.paymentTerm),

          // TVA fields (copy from quotation)
          totalBrutHT: quotation.totalBrutHT,
          totalDiscount: quotation.totalDiscount,
          totalNetHT: quotation.totalNetHT,
          tva19Amount: quotation.tva19Amount,
          tva7Amount: quotation.tva7Amount,
          tva0Amount: quotation.tva0Amount,
          totalTVA: quotation.totalTVA,
          netAPayer: Number(quotation.netAPayer) + 1.000,
          netAPayerWords: quotation.netAPayerWords,
          customerTvaExempt: quotation.customerTvaExempt,

          // Fiscal Snapshots (New Accounting Module)
          fiscalTotalHT: quotation.totalNetHT,
          fiscalTotalTVA: quotation.totalTVA,
          timbreFiscal: 1.000, // Default timbre
          matriculeFiscal: customer.taxId, // Use taxId as matricule fiscal

          // Deprecated (backward compatibility)
          subtotal: quotation.subtotal,
          taxAmount: quotation.taxAmount,
          totalAmount: Number(quotation.totalAmount) + 1.000,

          quotationId: quotation.id,
          notes: quotation.notes,
        },
      });

      for (const item of quotation.items) {
        await tx.orderItem.create({
          data: {
            orderId: order.id,
            productId: item.productId,
            productName: item.productName,
            productSku: item.productSku,
            selectedSize: item.selectedSize,
            selectedUnitType: item.selectedUnitType,
            quantity: item.quantity,
            unitPrice: item.unitPrice,

            // TVA fields (copy from quotation item)
            discount: item.discount,
            totalHT: item.totalHT,
            tvaRate: item.tvaRate,
            tvaAmount: item.tvaAmount,
            totalTTC: item.totalTTC,
            tvaExempt: item.tvaExempt,

            // Deprecated
            totalPrice: item.totalPrice,
          },
        });

        await tx.product.update({
          where: { id: item.productId },
          data: { stockQuantity: { decrement: item.quantity } },
        });
      }

      await tx.orderStatusHistory.create({
        data: {
          orderId: order.id,
          status: 'PENDING',
          notes: 'Order created from approved quotation',
        },
      });

      await tx.quotation.update({
        where: { id },
        data: {
          status: 'CONVERTED_TO_ORDER',
          convertedToOrderId: order.id,
          reviewedBy: adminUserId,
          reviewedAt: new Date(),
        },
      });

      await tx.user.update({
        where: { id: customer.id },
        data: { currentOutstanding: { increment: Number(quotation.netAPayer) } },
      });

      return { order, creditWarning };
    });

    res.json({
      success: true,
      data: result,
      message: result.creditWarning ? 'Order created - Customer exceeds credit limit' : 'Order created successfully',
    });
  } catch (error) {
    console.error('Error approving quotation:', error);
    // Log full error for debugging
    if (error instanceof Error) {
      console.error(error.stack);
    }
    res.status(500).json({ success: false, error: { message: 'Failed to approve quotation' } });
  }
};

export const declineQuotation = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const adminUserId = req.user?.id;
    const userRole = req.user?.role;
    const { reason } = req.body;

    if (userRole !== 'ADMIN' && userRole !== 'MANAGER') {
      return res.status(403).json({ success: false, error: { message: 'Forbidden' } });
    }

    if (!reason?.trim()) {
      return res.status(400).json({ success: false, error: { message: 'Decline reason required' } });
    }

    const quotation = await prisma.quotation.findUnique({ where: { id } });
    if (!quotation) {
      return res.status(404).json({ success: false, error: { message: 'Quotation not found' } });
    }

    if (quotation.status !== 'PENDING_APPROVAL') {
      return res.status(400).json({ success: false, error: { message: 'Only pending quotations can be declined' } });
    }

    const updated = await prisma.quotation.update({
      where: { id },
      data: {
        status: 'DECLINED',
        adminDecisionReason: reason,
        reviewedBy: adminUserId,
        reviewedAt: new Date(),
      },
      include: { user: true, reviewer: true },
    });

    res.json({ success: true, data: updated, message: 'Quotation declined' });
  } catch (error) {
    console.error('Error declining quotation:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to decline quotation' } });
  }
};

export const deleteQuotation = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id;

    const quotation = await prisma.quotation.findUnique({ where: { id } });
    if (!quotation) {
      return res.status(404).json({ success: false, error: { message: 'Quotation not found' } });
    }

    if (quotation.userId !== userId) {
      return res.status(403).json({ success: false, error: { message: 'Forbidden' } });
    }

    if (quotation.status !== 'DRAFT') {
      return res.status(400).json({ success: false, error: { message: 'Only drafts can be deleted' } });
    }

    await prisma.quotation.delete({ where: { id } });
    res.json({ success: true, message: 'Quotation deleted' });
  } catch (error) {
    console.error('Error deleting quotation:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to delete quotation' } });
  }
};
