import React, { useState, useEffect } from 'react';
import { Plus, Trash2, Edit2, Shield } from 'lucide-react';
import { buildAuthHeaders } from '../utils/apiClient';
import { useAuth, UserRole } from '../contexts/AuthContext';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

interface AdminUser {
    id: string;
    email: string;
    firstName: string;
    lastName: string;
    role: UserRole;
    isActive: boolean;
    createdAt: string;
}

export default function TeamPage() {
    const [admins, setAdmins] = useState<AdminUser[]>([]);
    const [isLoading, setIsLoading] = useState(true);
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editingAdmin, setEditingAdmin] = useState<AdminUser | null>(null);
    const { hasRole } = useAuth();

    // Form state
    const [formData, setFormData] = useState({
        email: '',
        password: '',
        firstName: '',
        lastName: '',
        role: UserRole.SALES_ADMIN,
        phone: ''
    });

    useEffect(() => {
        fetchAdmins();
    }, []);

    const fetchAdmins = async () => {
        try {
            const response = await fetch(`${API_URL}/admin/users`, {
                headers: buildAuthHeaders()
            });
            if (response.ok) {
                const data = await response.json();
                setAdmins(data.data);
            }
        } catch (error) {
            console.error('Failed to fetch admins', error);
        } finally {
            setIsLoading(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            const url = editingAdmin
                ? `${API_URL}/admin/users/${editingAdmin.id}/role` // Only role update for edit in this simple version
                : `${API_URL}/admin/users`;

            const method = editingAdmin ? 'PUT' : 'POST';
            const body = editingAdmin
                ? JSON.stringify({ role: formData.role })
                : JSON.stringify(formData);

            const response = await fetch(url, {
                method,
                headers: buildAuthHeaders(),
                body
            });

            if (response.ok) {
                setIsModalOpen(false);
                setEditingAdmin(null);
                setFormData({
                    email: '',
                    password: '',
                    firstName: '',
                    lastName: '',
                    role: UserRole.SALES_ADMIN,
                    phone: ''
                });
                fetchAdmins();
            }
        } catch (error) {
            console.error('Failed to save admin', error);
        }
    };

    const handleDelete = async (id: string) => {
        if (!window.confirm('Are you sure you want to delete this admin?')) return;

        try {
            const response = await fetch(`${API_URL}/admin/users/${id}`, {
                method: 'DELETE',
                headers: buildAuthHeaders()
            });

            if (response.ok) {
                fetchAdmins();
            }
        } catch (error) {
            console.error('Failed to delete admin', error);
        }
    };

    const openEdit = (admin: AdminUser) => {
        setEditingAdmin(admin);
        setFormData({
            email: admin.email,
            password: '', // Password not editable here
            firstName: admin.firstName,
            lastName: admin.lastName,
            role: admin.role,
            phone: ''
        });
        setIsModalOpen(true);
    };

    if (!hasRole([UserRole.SUPER_ADMIN, UserRole.ADMIN])) {
        return <div className="p-8 text-center text-gray-500">You do not have permission to view this page.</div>;
    }

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <h1 className="text-2xl font-bold text-gray-900">Admin Team</h1>
                <button
                    onClick={() => {
                        setEditingAdmin(null);
                        setFormData({ ...formData, email: '', password: '' });
                        setIsModalOpen(true);
                    }}
                    className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                    <Plus className="w-4 h-4" />
                    Add New Admin
                </button>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                <table className="w-full text-left">
                    <thead className="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th className="px-6 py-4 font-medium text-gray-500">Name</th>
                            <th className="px-6 py-4 font-medium text-gray-500">Email</th>
                            <th className="px-6 py-4 font-medium text-gray-500">Role</th>
                            <th className="px-6 py-4 font-medium text-gray-500">Status</th>
                            <th className="px-6 py-4 font-medium text-gray-500 text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-200">
                        {admins.map((admin) => (
                            <tr key={admin.id} className="hover:bg-gray-50">
                                <td className="px-6 py-4">
                                    <div className="font-medium text-gray-900">{admin.firstName} {admin.lastName}</div>
                                </td>
                                <td className="px-6 py-4 text-gray-600">{admin.email}</td>
                                <td className="px-6 py-4">
                                    <span className="inline-flex items-center gap-1.5 px-2.5 py-1 rounded-full text-xs font-medium bg-purple-100 text-purple-700">
                                        <Shield className="w-3 h-3" />
                                        {admin.role.replace('_', ' ')}
                                    </span>
                                </td>
                                <td className="px-6 py-4">
                                    <span className={`inline-flex items-center px-2.5 py-1 rounded-full text-xs font-medium ${admin.isActive ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'
                                        }`}>
                                        {admin.isActive ? 'Active' : 'Inactive'}
                                    </span>
                                </td>
                                <td className="px-6 py-4 text-right space-x-2">
                                    <button
                                        onClick={() => openEdit(admin)}
                                        className="p-1 text-gray-400 hover:text-blue-600 transition-colors"
                                    >
                                        <Edit2 className="w-4 h-4" />
                                    </button>
                                    <button
                                        onClick={() => handleDelete(admin.id)}
                                        className="p-1 text-gray-400 hover:text-red-600 transition-colors"
                                    >
                                        <Trash2 className="w-4 h-4" />
                                    </button>
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>

            {isModalOpen && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
                    <div className="bg-white rounded-xl shadow-xl max-w-md w-full p-6">
                        <h2 className="text-xl font-bold mb-4">{editingAdmin ? 'Edit Admin Role' : 'New Admin'}</h2>
                        <form onSubmit={handleSubmit} className="space-y-4">
                            {!editingAdmin && (
                                <>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">First Name</label>
                                            <input
                                                required
                                                className="w-full px-3 py-2 border rounded-lg"
                                                value={formData.firstName}
                                                onChange={e => setFormData({ ...formData, firstName: e.target.value })}
                                            />
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">Last Name</label>
                                            <input
                                                required
                                                className="w-full px-3 py-2 border rounded-lg"
                                                value={formData.lastName}
                                                onChange={e => setFormData({ ...formData, lastName: e.target.value })}
                                            />
                                        </div>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">Email</label>
                                        <input
                                            required
                                            type="email"
                                            className="w-full px-3 py-2 border rounded-lg"
                                            value={formData.email}
                                            onChange={e => setFormData({ ...formData, email: e.target.value })}
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">Password</label>
                                        <input
                                            required
                                            type="password"
                                            className="w-full px-3 py-2 border rounded-lg"
                                            value={formData.password}
                                            onChange={e => setFormData({ ...formData, password: e.target.value })}
                                        />
                                    </div>
                                </>
                            )}

                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">Role</label>
                                <select
                                    className="w-full px-3 py-2 border rounded-lg"
                                    value={formData.role}
                                    onChange={e => setFormData({ ...formData, role: e.target.value as UserRole })}
                                >
                                    <option value={UserRole.SUPER_ADMIN}>Super Admin</option>
                                    <option value={UserRole.SALES_ADMIN}>Sales Admin</option>
                                    <option value={UserRole.CATALOG_ADMIN}>Catalog Admin</option>
                                    <option value={UserRole.ACCOUNTING_ADMIN}>Accounting Admin</option>
                                    <option value={UserRole.SUPPORT_ADMIN}>Support Admin</option>
                                    <option value={UserRole.MANAGER}>Manager</option>
                                </select>
                            </div>

                            <div className="flex justify-end gap-3 mt-6">
                                <button
                                    type="button"
                                    onClick={() => setIsModalOpen(false)}
                                    className="px-4 py-2 text-gray-600 hover:bg-gray-100 rounded-lg"
                                >
                                    Cancel
                                </button>
                                <button
                                    type="submit"
                                    className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                                >
                                    {editingAdmin ? 'Update Role' : 'Create Admin'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
